"use-client";
import React, { useEffect, useState } from "react";
import Image from "next/image";

// Define the possible outcomes
export type Outcome = "PLAYER" | "BANKER" | "PANDA" | "DRAGON" | "TIE";

// Props for the BottomGrid component
interface BottomGridProps {
  results: Outcome[]; // Array of game results
  rows?: number; // Number of rows in the grid
  cols?: number; // Number of columns in the grid
  displayColumns?: number; // How many recent columns to show
}

// Default constants for main outcomes
const PLAYER_MAIN_OUTCOME: Outcome = "PLAYER";
const BANKER_MAIN_OUTCOME: Outcome = "BANKER";

export const BottomGridBoard: React.FC<BottomGridProps> = ({
  results,
  rows = 6,
  cols = 36,
  displayColumns = 14,
}) => {
  const [isMobile, setIsMobile] = useState(false);

  useEffect(() => {
    const checkScreenSize = () => {
      setIsMobile(window.innerWidth < 640);
    };

    checkScreenSize(); // initial
    window.addEventListener("resize", checkScreenSize);
    return () => window.removeEventListener("resize", checkScreenSize);
  }, []);

  const boxWidth = isMobile ? "8.3vw" : "45.5px";
  const boxHeight = isMobile ? "8.3vw" : "45.5px";

  // Initialize empty grid
  const grid: (React.ReactNode | null)[][] = Array(rows)
    .fill(null)
    .map(() => Array(cols).fill(null));

  let currentCol = 0;
  let currentRow = 0;
  let lastOutcomeGroup: string | null = null;

  const isBankerGroup = (res: Outcome) =>
    res === BANKER_MAIN_OUTCOME || res === "DRAGON" || res === "TIE"; // TIE now belongs to BANKER group
  const isPlayerGroup = (res: Outcome) =>
    res === PLAYER_MAIN_OUTCOME || res === "PANDA";
  const getOutcomeGroup = (res: Outcome) => {
    if (isBankerGroup(res)) return "banker_group";
    if (isPlayerGroup(res)) return "player_group";
    return res; // This case should no longer be reached since TIE is now in banker_group
  };

  // Fill the grid based on outcomes
  results.forEach((result) => {
    let imageNode: React.ReactNode = null;
    switch (result) {
      case "PLAYER":
        imageNode = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/player.png"
              alt="Player"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "BANKER":
        imageNode = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/banker.png"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "PANDA":
        imageNode = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/pandaGIF.gif"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "DRAGON":
        imageNode = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/dragonGIF.gif"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "TIE":
        imageNode = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/TIEGif2.gif"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
    }

    const group = getOutcomeGroup(result);
    if (lastOutcomeGroup && lastOutcomeGroup !== group) {
      currentCol++;
      currentRow = 0;
    }
    lastOutcomeGroup = group;

    if (currentRow < rows && currentCol < cols) {
      grid[currentRow][currentCol] = imageNode;
      currentRow++;
    } else {
      // Overflow logic: place in bottom row
      let r = rows - 1;
      let c = currentCol;
      while (c < cols - 1 && grid[r][c] !== null) c++;
      if (grid[r][c] === null) {
        grid[r][c] = imageNode;
      } else {
        currentCol++;
        currentRow = 0;
        if (currentCol < cols) {
          grid[currentRow][currentCol] = imageNode;
          currentRow++;
        }
      }
    }
  });

  // Determine visible columns
  let maxColWithContent: number = grid[0].reduceRight((acc: number, _, idx) => {
    const has = grid.some((row) => row[idx] !== null);
    return has && acc === -1 ? idx : acc;
  }, -1);
  maxColWithContent = Math.max(0, maxColWithContent);
  const startCol = Math.max(0, maxColWithContent - displayColumns + 1);

  // Render the grid items
  const items: React.ReactNode[] = [];
  for (let r = 0; r < rows; r++) {
    for (let c = 0; c < displayColumns; c++) {
      const colIndex = startCol + c;
      items.push(
        <div
          key={`cell-${r}-${colIndex}`}
          className="border border-[#675e57] flex items-center justify-center"
          style={{
            backgroundColor: "#f5e9d0",
            width: boxWidth,
            height: boxHeight,
          }}
        >
          {colIndex < cols ? grid[r][colIndex] : null}
        </div>
      );
    }
  }

  return (
    <div className="w-full overflow-hidden">
      <div
        className="grid grid-rows-6 gap-0 ml-auto"
        style={{
          gridTemplateColumns: `repeat(${displayColumns}, ${boxWidth})`,
          width: isMobile ? "100vw" : `${displayColumns * 45.5}px`,
        }}
      >
        {items}
      </div>
    </div>
  );
};
